#region References

using System;
using System.IO;
using System.Text;
using System.Data;
using System.Collections;
using gov.va.med.vbecs.Common.Log;

#endregion

namespace gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>David Askew</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>5/23/2008</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary>The HL7BtsMessage type represents the HL7 v2.5 Blood Product Transfusion/Disposition message. It implements the HL7ProtocolMessage type and is not meant to be inherited.</summary>

	#endregion 

	/// <summary>
	/// New class for Patient Updates 
	/// CR 2985
	/// </summary>
	public sealed class HL7AdtMessage : HL7ProtocolMessage
	{
		#region Variables

        // Events Logger
        private readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

		/// <summary>
		/// Accepted Message Type
		/// </summary>
		private const string ADT = "ADT";

		/// <summary>
		/// Accepted Trigger Events
		/// </summary>
		private const string A01 = "A01";
		private const string A02 = "A02";
		private const string A03 = "A03";
		private const string A04 = "A04";
		private const string A08 = "A08";
		private const string A11 = "A11";
		private const string A12 = "A12";
		private const string A13 = "A13";

		private const bool RESPONSE_REQUIRED_INDICATOR = true;

		private char [] _delimiters;

		private string _triggerEvent;

		private string[] _msh;
		private string[] _evn;
		private string[] _pid;
		private string[] _pv1;

		/// <summary>
		/// Use to indicate any error(s) encountered while loading the message.  These codes  
		/// will be returned in a negative acknowledgement to the sending application.
		/// </summary>
		AckErrorCodes _ackErrorCode;

		#endregion

		#region Constructors

		/// <summary>
		/// Empty static constructor added to get rid of "beforefieldinit" attribute generated by compiler.
		/// </summary>
		static HL7AdtMessage() {}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8863"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8864"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// 
		/// </summary>
		/// <param name="message"></param>
		public HL7AdtMessage( string message )
			: base( message )
		{
			if( message == null )
			{
				throw( new ArgumentNullException( "message" ) );
			}
			//
			_ackErrorCode = AckErrorCodes.MessageAccepted;
			//
			_message = message;
			//
			try
			{
				// This includes field separator in position [0] and encoding characters in [1] - [4]
				_delimiters = HL7Utility.ParseGetMessageDelimiters(message);			
				//
				// Split the message into segments
				LoadDataFromHL7Message();
				//
				_messageControlID = HL7Utility.GetMessageControlID( message );
				//
				string [] messageTypeAndTriggerEvent = MSH[9].Split(_delimiters[1]);
				//
				if ( messageTypeAndTriggerEvent != null && messageTypeAndTriggerEvent.Length == 2 )
				{
					_messageType = HL7Utility.ConvertString( messageTypeAndTriggerEvent[0] );
					//
					_triggerEvent = HL7Utility.ConvertString( messageTypeAndTriggerEvent[1] );
				}
			}
			catch(Exception ex)
			{
				_ackErrorCode = AckErrorCodes.SegmentSequenceError;
				//
				var innerException = HL7Utility.GetInnerExceptionFromException(ex);
				//
                // EventLogAppender: note that I added reference to VBECS.Common
                _eventsLogger.Error("Error processing HL7AdtMessage.\n\n" + innerException);
                //System.Diagnostics.EventLog.WriteEntry( "Error processing HL7AdtMessage.\n\n", innerException, System.Diagnostics.EventLogEntryType.Error );
			}
		}

		#endregion 

		#region Methods

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8867"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8868"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the full HL7 message.
		/// </summary>
		/// <returns>string representation of HL7 message input.</returns>
		public override string GetMessage()
		{
			return Message;
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8869"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8870"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the Message Control ID associated with this message
		/// </summary>
		/// <returns>Message Control ID in MSH segment</returns>
		public override string GetMessageControlID()
		{
			return MessageControlID;
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8871"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8872"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Overriden method to get the Message Type.
		/// </summary>
		/// <returns>Message Type, with Trigger Event, in MSH segment.</returns>
		public override string GetMessageType()
		{
			return MessageType;
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8873"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8874"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Currently only ADT A04 and A08 messages 
		/// include updated patient demographic data
		/// CR 2979
		/// </summary>
		public bool IsDemographicUpdate()
		{
			switch( _triggerEvent )
			{
				case A04:
				case A08:
					return true;
				default:
					return false;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/5/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8901"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8902"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// 
		/// </summary>
		public bool IsDischarge()
		{
			switch( _triggerEvent )
			{
				case A03:
					return true;
				default:
					return false;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8875"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8876"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// 
		/// </summary>
		public bool IsMessageTypeValid()
		{
			switch( _messageType )
			{
				case ADT:
					return true;
				default:
					return false;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/12/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8908"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8909"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// 
		/// </summary>
		public bool IsTriggerEventValid()
		{
			switch( _triggerEvent )
			{
				case A01:
				case A02:
				case A03:
				case A04:
				case A08:
				case A11:
				case A12:
				case A13:
					return true;
				default:
					return false;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8865"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8866"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method always returns true for this type of message.
		/// </summary>
		/// <returns>boolean</returns>
		public override bool IsResponseRequired()
		{
			return RESPONSE_REQUIRED_INDICATOR;
		}

		/// <summary>
		/// WriteHL7MessageBody
		/// </summary>
		/// <param name="writer"></param>
		protected override void WriteHL7MessageBody( StreamWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.Write( this.Message );
		}

		/// <summary>
		/// 
		/// </summary>
		private void LoadDataFromHL7Message()
		{
			string [] hl7Segments = HL7Utility.ParseGetAllMessageSegments( this.Message );
			//
			for (int i = 0 ; i < hl7Segments.Length - 1 ; i++ )
			{
				string [] seg = hl7Segments[i].Split(_delimiters[0]);
				//
				switch( seg[0] )
				{
					case "MSH":
					{
						string [] mshTemp = hl7Segments[i].Split(_delimiters[0]);	
						_msh = new string[mshTemp.Length + 1];
						_msh[0] = mshTemp[0];
						_msh[1] = _delimiters[0].ToString();
						for (int j = 2; j < mshTemp.Length; j++)
						{
							_msh[j] =  mshTemp[j - 1];
						}
						break;
					}
					case "EVN":
					{
						_evn = hl7Segments[i].Split(_delimiters[0]);
						break; 
					}
					case "PID":
					{
						_pid = hl7Segments[i].Split(_delimiters[0]);
						break; 
					}
					case "PV1":
					{
						_pv1 = hl7Segments[i].Split(_delimiters[0]);
						break; 
					}
					default:
					{
						break;
					}
				}
			}
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Hines OIFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/30/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9198"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9199"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// AckErrorCode
		/// </summary>
		public AckErrorCodes AckErrorCode
		{
			get
			{
				return _ackErrorCode;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8883"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8884"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// 
		/// </summary>
		public char[] Delimiters
		{
			get
			{
				return _delimiters;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8877"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8878"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MessageType
		/// </summary>
		public string MessageType
		{
			get
			{
				return _messageType;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8889"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8890"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// 
		/// </summary>
		public string TriggerEvent
		{
			get
			{
				return _triggerEvent;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8879"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8880"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Full HL7 message
		/// </summary>
		public string Message
		{
			get
			{
				return _message;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8881"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8882"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Message Control ID
		/// </summary>
		public string MessageControlID
		{
			get
			{
				return _messageControlID;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8885"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8886"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MSH segment
		/// </summary>
		public string [] MSH
		{
			get
			{
				return _msh;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8895"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8896"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// EVN segment
		/// </summary>
		public string [] EVN
		{
			get
			{
				return _evn;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8893"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8894"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PID segment
		/// </summary>
		public string [] PID
		{
			get
			{
				return _pid;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8891"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8892"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PV1 segment
		/// </summary>
		public string [] PV1
		{
			get
			{
				return _pv1;
			}
		}
		
		#endregion
	}
}
